//
//  FFAssetExportSession.h
//  LibAVFoundation
//
//  Created by Jörgen Isaksson on 2014-03-14.
//  Copyright (c) 2014 Bitfield AB. All rights reserved.
//

#import <Foundation/Foundation.h>

@class FFAsset;

enum {
	FFAssetExportSessionStatusUnknown,
    FFAssetExportSessionStatusWaiting,
    FFAssetExportSessionStatusExporting,
    FFAssetExportSessionStatusCompleted,
    FFAssetExportSessionStatusFailed,
    FFAssetExportSessionStatusCancelled
};
typedef NSInteger FFAssetExportSessionStatus;

@interface FFAssetExportSession : NSObject

+ (NSArray *)allExportPresets;

/* Indicates the name of the preset with which the AVExportSession was initialized */
@property (nonatomic, readonly) NSString *presetName;

/* Indicates the instance of AVAsset with which the AVExportSession was initialized  */
@property (nonatomic, retain, readonly) FFAsset *asset;

/* Indicates the type of file to be written by the session.
 The value of this property must be set before you invoke -exportAsynchronouslyWithCompletionHandler:; otherwise -exportAsynchronouslyWithCompletionHandler: will raise an NSInternalInconsistencyException.
 Setting the value of this property to a file type that's not among the session's supported file types will result in an NSInvalidArgumentException. See supportedFileTypes. */
@property (nonatomic, copy) NSString *outputFileType;

/* Indicates the URL of the export session's output.  */
@property (nonatomic, copy) NSURL *outputURL;

/* indicates the status of the export session */
@property (nonatomic, readonly) FFAssetExportSessionStatus status;

/* describes the error that occured if the export status is AVAssetExportSessionStatusFailed */
@property (nonatomic, readonly) NSError *error;

/* Specifies the progress of the export on a scale from 0 to 1.0.  A value of 0 means the export has not yet begun, A value of 1.0 means the export is complete. This property is not key-value observable. */
@property (nonatomic, readonly) float progress;

/* Specifies a time range to be exported from the source.  The default timeRange of an export session is kCMTimeZero..kCMTimePositiveInfinity, meaning that the full duration of the asset will be exported. */
@property (nonatomic) CMTimeRange timeRange;

/* Specifies an NSArray of AVMetadataItems that are to be written to the output file by the export session.
 If the value of this key is nil, any existing metadata in the exported asset will be translated as accurately as possible into
 the appropriate metadata keyspace for the output file and written to the output. */
@property (nonatomic, copy) NSArray *metadata;

/* indicates the movie should be optimized for network use */
@property (nonatomic) BOOL shouldOptimizeForNetworkUse;

- (id)initWithAsset:(FFAsset *)asset presetName:(NSString *)presetName error:(NSError **)outError;

/*!
	@method						exportAsynchronouslyWithCompletionHandler:
	@abstract					Starts the asynchronous execution of an export session.
	@param						handler
								If internal preparation for export fails, the handler will be invoked synchronously.
								The handler may also be called asynchronously after -exportAsynchronouslyWithCompletionHandler: returns,
								in the following cases: 
								1) if a failure occurs during the export, including failures of loading, re-encoding, or writing media data to the output,
								2) if -cancelExport is invoked, 
								3) if export session succeeds, having completely written its output to the outputURL. 
								In each case, AVAssetExportSession.status will signal the terminal state of the asset reader, and if a failure occurs, the NSError 
								that describes the failure can be obtained from the error property.
	@discussion					Initiates an asynchronous export operation and returns immediately.
*/
- (void)exportAsynchronouslyWithCompletionHandler:(void (^)(void))handler;

/*!
	@method						cancelExport
	@abstract					Cancels the execution of an export session.
	@discussion					Cancel can be invoked when the export is running.
*/
- (void)cancelExport;

@end
